# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Custom content negotiation for package archives."""

from collections.abc import Iterable

from rest_framework.negotiation import BaseContentNegotiation
from rest_framework.renderers import BaseRenderer
from rest_framework.request import Request


# Based on: https://www.django-rest-framework.org/api-guide/content-negotiation/
class IgnoreClientContentNegotiation(BaseContentNegotiation):
    """
    Ignore client content negotiation attempts.

    Since many of these views return files rather than anything that can be
    rendered differently, they can't do anything useful in response to
    content negotiation.  However, they need to not return 406 Not
    Acceptable if the client sends an ``Accept`` header.
    """

    def select_renderer(
        self,
        request: Request,  # noqa: U100
        renderers: Iterable[BaseRenderer],
        format_suffix: str | None = None,
    ) -> tuple[BaseRenderer, str]:
        """Select any renderer."""
        format_suffix  # fake usage for vulture
        renderer = next(iter(renderers))
        return (renderer, renderer.media_type)
