// SPDX-License-Identifier: LGPL-2.1-or-later
//
// SPDX-FileCopyrightText: 2012 Mohammed Nafees <nafees.technocool@gmail.com>
//

#include "GeoDataListStyle.h"
#include "GeoDataItemIcon.h"
#include "GeoDataTypes.h"
#include "MarbleDirs.h"

#include <QDataStream>

namespace Marble
{

class GeoDataListStylePrivate
{
public:
    GeoDataListStylePrivate();

    GeoDataListStyle::ListItemType m_listItemType;
    QColor m_bgColor;

    QList<GeoDataItemIcon *> m_vector;
};

GeoDataListStylePrivate::GeoDataListStylePrivate()
    : m_listItemType(GeoDataListStyle::Check)
    , m_bgColor(Qt::white)
{
}

GeoDataListStyle::GeoDataListStyle()
    : d(new GeoDataListStylePrivate)
{
}

GeoDataListStyle::GeoDataListStyle(const Marble::GeoDataListStyle &other)
    : GeoDataObject(other)
    , d(new GeoDataListStylePrivate(*other.d))
{
}

GeoDataListStyle &GeoDataListStyle::operator=(const GeoDataListStyle &other)
{
    GeoDataObject::operator=(other);
    *d = *other.d;
    return *this;
}

bool GeoDataListStyle::operator==(const GeoDataListStyle &other) const
{
    if (!GeoDataObject::equals(other) || d->m_bgColor != other.d->m_bgColor || d->m_listItemType != other.d->m_listItemType
        || d->m_vector.size() != other.d->m_vector.size()) {
        return false;
    }

    QList<GeoDataItemIcon *>::const_iterator begin = d->m_vector.constBegin();
    QList<GeoDataItemIcon *>::const_iterator end = d->m_vector.constEnd();
    QList<GeoDataItemIcon *>::const_iterator otherBegin = other.d->m_vector.constBegin();

    for (; begin != end; ++begin, ++otherBegin) {
        if (**begin != **otherBegin) {
            return false;
        }
    }

    return true;
}

bool GeoDataListStyle::operator!=(const GeoDataListStyle &other) const
{
    return !this->operator==(other);
}

GeoDataListStyle::~GeoDataListStyle()
{
    delete d;
}

const char *GeoDataListStyle::nodeType() const
{
    return GeoDataTypes::GeoDataListStyleType;
}

GeoDataListStyle::ListItemType GeoDataListStyle::listItemType() const
{
    return d->m_listItemType;
}

void GeoDataListStyle::setListItemType(ListItemType type)
{
    d->m_listItemType = type;
}

QColor GeoDataListStyle::backgroundColor() const
{
    return d->m_bgColor;
}

void GeoDataListStyle::setBackgroundColor(const QColor &color)
{
    d->m_bgColor = color;
}

QList<GeoDataItemIcon *> GeoDataListStyle::itemIconList() const
{
    return d->m_vector;
}

GeoDataItemIcon *GeoDataListStyle::child(int i)
{
    return d->m_vector.at(i);
}

const GeoDataItemIcon *GeoDataListStyle::child(int i) const
{
    return d->m_vector.at(i);
}

int GeoDataListStyle::childPosition(const GeoDataItemIcon *object) const
{
    return d->m_vector.indexOf(const_cast<GeoDataItemIcon *>(object));
}

void GeoDataListStyle::append(GeoDataItemIcon *other)
{
    other->setParent(this);
    d->m_vector.append(other);
}

void GeoDataListStyle::remove(int index)
{
    d->m_vector.remove(index);
}

int GeoDataListStyle::size() const
{
    return d->m_vector.size();
}

GeoDataItemIcon &GeoDataListStyle::at(int pos)
{
    return *(d->m_vector[pos]);
}

const GeoDataItemIcon &GeoDataListStyle::at(int pos) const
{
    return *(d->m_vector.at(pos));
}

GeoDataItemIcon &GeoDataListStyle::last()
{
    return *(d->m_vector.last());
}

const GeoDataItemIcon &GeoDataListStyle::last() const
{
    return *(d->m_vector.last());
}

GeoDataItemIcon &GeoDataListStyle::first()
{
    return *(d->m_vector.first());
}

const GeoDataItemIcon &GeoDataListStyle::first() const
{
    return *(d->m_vector.first());
}

void GeoDataListStyle::clear()
{
    qDeleteAll(d->m_vector);
    d->m_vector.clear();
}

QList<GeoDataItemIcon *>::Iterator GeoDataListStyle::begin()
{
    return d->m_vector.begin();
}

QList<GeoDataItemIcon *>::Iterator GeoDataListStyle::end()
{
    return d->m_vector.end();
}

QList<GeoDataItemIcon *>::ConstIterator GeoDataListStyle::constBegin() const
{
    return d->m_vector.constBegin();
}

QList<GeoDataItemIcon *>::ConstIterator GeoDataListStyle::constEnd() const
{
    return d->m_vector.constEnd();
}

void GeoDataListStyle::pack(QDataStream &stream) const
{
    GeoDataObject::pack(stream);
    stream << d->m_vector.count();

    for (QList<GeoDataItemIcon *>::const_iterator iterator = d->m_vector.constBegin(); iterator != d->m_vector.constEnd(); ++iterator) {
        const GeoDataItemIcon *itemIcon = *iterator;
        itemIcon->pack(stream);
    }
}

void GeoDataListStyle::unpack(QDataStream &stream)
{
    GeoDataObject::unpack(stream);

    int count;
    stream >> count;

    int featureId;
    stream >> featureId;

    auto itemIcon = new GeoDataItemIcon;
    itemIcon->unpack(stream);
    d->m_vector.append(itemIcon);
}

}
